<?php

namespace App\Http\Requests\DailyPatrolSummary;

use App\Http\Requests\BaseRequest;
use App\Enums\DailyPatrolSummaryStatus;
use Illuminate\Validation\Rule;

class StoreDailyPatrolSummaryRequest extends BaseRequest
{
    public function authorize(): bool
    {
        return true;
    }

    public function rules(): array
    {
        return [
            'guard_id' => ['required', 'exists:guards,id'],
            'patrol_round_id' => ['required', 'exists:patrol_rounds,id'],
            'patrol_date' => ['required', 'date'],
            'total_locations' => ['required', 'integer', 'min:0'],
            'completed_locations' => ['required', 'integer', 'min:0', 'lte:total_locations'],
            'missed_locations' => ['required', 'integer', 'min:0'],
            'expected_start_time' => ['required', 'date_format:H:i'],
            'expected_end_time' => ['required', 'date_format:H:i', 'after:expected_start_time'],
            'actual_start_time' => ['nullable', 'date'],
            'actual_end_time' => ['nullable', 'date', 'after:actual_start_time'],
            'status' => ['required', Rule::enum(DailyPatrolSummaryStatus::class)],
            'notes' => ['nullable', 'string', 'max:1000'],
        ];
    }

    public function messages(): array
    {
        return [
            'guard_id.required' => 'Guard is required',
            'guard_id.exists' => 'Selected guard does not exist',
            'patrol_round_id.required' => 'Patrol round is required',
            'patrol_round_id.exists' => 'Selected patrol round does not exist',
            'patrol_date.required' => 'Patrol date is required',
            'patrol_date.date' => 'Patrol date must be a valid date',
            'total_locations.required' => 'Total locations is required',
            'total_locations.min' => 'Total locations must be at least 0',
            'completed_locations.required' => 'Completed locations is required',
            'completed_locations.lte' => 'Completed locations cannot exceed total locations',
            'expected_start_time.required' => 'Expected start time is required',
            'expected_start_time.date_format' => 'Expected start time must be in HH:MM format',
            'expected_end_time.required' => 'Expected end time is required',
            'expected_end_time.date_format' => 'Expected end time must be in HH:MM format',
            'expected_end_time.after' => 'Expected end time must be after start time',
            'actual_end_time.after' => 'Actual end time must be after actual start time',
            'status.required' => 'Status is required',
        ];
    }

    protected function prepareForValidation(): void
    {
        // Calculate missed locations if not provided
        if ($this->has('total_locations') && $this->has('completed_locations') && !$this->has('missed_locations')) {
            $this->merge([
                'missed_locations' => max(0, $this->total_locations - $this->completed_locations)
            ]);
        }
    }
}