<?php

namespace App\Http\Controllers\Api;

use App\Helpers\ApiResponse;
use App\Http\Controllers\Controller;
use App\Http\Requests\User\StoreUserRequest;
use App\Http\Requests\User\UpdateUserRequest;
use App\Http\Resources\UserResource;
use App\Models\User;
use App\Traits\CanManageFile;
use Illuminate\Http\Request;
use Spatie\QueryBuilder\QueryBuilder;

class UserController extends Controller
{
    use CanManageFile;

    public function __construct() {
        $this->resourceName = "users";
        $this->fileFieldNames = ["photo"];
        $this->fileFolderPaths = ["photos"];
    }

    public function index() {
        $query = QueryBuilder::for(User::class)
            ->allowedFilters(['owner_id', 'flat_id', 'name', 'email', 'phone', 'role', 'gender',])
            ->allowedIncludes(['owner', 'flat'])
            ->latest();

        $users = $query
            ->paginate(request("perPage", 15));

        return ApiResponse::paginated($users);
    }

    public function store(StoreUserRequest $request) {
        $user = new User($request->validated());
        $this->storeFiles($request, $user);
        $user->save();

        return ApiResponse::created(
            new UserResource($user),
            "User has been created successfully!"
        );
    }

    public function show(User $user) {

        return ApiResponse::resource(new UserResource($user));
    }

    public function update(UpdateUserRequest $request, User $user) {
        $user->fill($request->validated());
        $this->updateFiles($request, $user);
        $user->save();

        return ApiResponse::updated(
            new UserResource($user),
            "User has been updated successfully!"
        );
    }

    public function destroy(User $user) {
        $this->deleteFiles($user);
        $user->delete();

        return ApiResponse::deleted("User has been deleted successfully!");
    }

    public function deleteProfile(Request $request, User $user) {
        $this->deleteFiles($user);

        return ApiResponse::deleted("Profile picture has been deleted successfully!");
    }
}
