<?php

namespace Database\Factories;

use App\Enums\PatrollingStatus;
use App\Models\Guard;
use App\Models\PatrollingLocation;
use App\Models\PatrollingLog;
use Illuminate\Database\Eloquent\Factories\Factory;

class PatrollingLogFactory extends Factory
{
    protected $model = PatrollingLog::class;

    public function definition(): array
    {
        $patrolDate = $this->faker->dateTimeBetween('-30 days', 'now');
        $status = $this->faker->randomElement(PatrollingStatus::cases());
        
        return [
            'location_id' => PatrollingLocation::factory(),
            'guard_id' => Guard::factory(),
            'patrol_date' => $patrolDate->format('Y-m-d'),
            'patrol_time' => $this->faker->dateTimeBetween($patrolDate->format('Y-m-d'), $patrolDate->format('Y-m-d') . ' 23:59:59'),
            'status' => $status,
            'notes' => $this->faker->optional()->sentence(),
            'issues_found' => $this->getIssuesForStatus($status),
        ];
    }

    private function getIssuesForStatus(PatrollingStatus $status): ?string
    {
        return match ($status) {
            PatrollingStatus::ISSUE_FOUND => $this->faker->randomElement([
                'Broken lighting fixture',
                'Water leakage observed',
                'Damaged door lock',
                'Suspicious activity reported',
                'Equipment malfunction'
            ]),
            PatrollingStatus::MAINTENANCE_REQUIRED => $this->faker->randomElement([
                'Cleaning required',
                'Painting needed',
                'Equipment service due',
                'Garden maintenance required',
                'Safety equipment check needed'
            ]),
            PatrollingStatus::NOT_ACCESSIBLE => $this->faker->randomElement([
                'Area locked',
                'Construction in progress',
                'Maintenance work ongoing',
                'Area blocked due to event',
                'Safety hazard present'
            ]),
            PatrollingStatus::CHECKED => null,
        };
    }

    public function checked(): static
    {
        return $this->state(fn (array $attributes) => [
            'status' => PatrollingStatus::CHECKED,
            'issues_found' => null,
        ]);
    }

    public function issueFound(): static
    {
        return $this->state(fn (array $attributes) => [
            'status' => PatrollingStatus::ISSUE_FOUND,
            'issues_found' => $this->getIssuesForStatus(PatrollingStatus::ISSUE_FOUND),
        ]);
    }

    public function maintenanceRequired(): static
    {
        return $this->state(fn (array $attributes) => [
            'status' => PatrollingStatus::MAINTENANCE_REQUIRED,
            'issues_found' => $this->getIssuesForStatus(PatrollingStatus::MAINTENANCE_REQUIRED),
        ]);
    }

    public function notAccessible(): static
    {
        return $this->state(fn (array $attributes) => [
            'status' => PatrollingStatus::NOT_ACCESSIBLE,
            'issues_found' => $this->getIssuesForStatus(PatrollingStatus::NOT_ACCESSIBLE),
        ]);
    }

    public function today(): static
    {
        return $this->state(fn (array $attributes) => [
            'patrol_date' => now()->format('Y-m-d'),
            'patrol_time' => $this->faker->dateTimeBetween('today', 'now'),
        ]);
    }

    public function yesterday(): static
    {
        return $this->state(fn (array $attributes) => [
            'patrol_date' => now()->subDay()->format('Y-m-d'),
            'patrol_time' => $this->faker->dateTimeBetween('yesterday', 'yesterday 23:59:59'),
        ]);
    }
}